#include <math.h>
#include <sqlite3.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <openssl/sha.h>
#include <string.h>

void sqlite_fail_on_err(sqlite3 *db) {
  if (sqlite3_errcode(db) == SQLITE_OK) return;
  fprintf(stderr, "SQLite Error: %s\n", sqlite3_errmsg(db));
  exit(1);
} 

void sqlite_conn_destr(sqlite3 **db) {
  sqlite3_close_v2(*db);
}

void sqlite_stmt_destr(sqlite3_stmt **stmt) { sqlite3_finalize(*stmt); }

void mem_destr(void *p) { free(*(void**)p); }

unsigned _BitInt(256) sha256hash(const char *input) {
  unsigned _BitInt(256) ret;
  unsigned char md[SHA256_DIGEST_LENGTH];
  SHA256((unsigned char*)input, strlen(input), md);
  for (size_t i = 0; i < SHA256_DIGEST_LENGTH; ++i)
    ret <<= 8, ret |= md[i];
  return ret;
}

int main(int argc, const char **argv) {
  if (argc != 3) {
    fprintf(stderr, "invalid number of arguments\n");
    return 0;
  }

  int roundnum = atoi(argv[1]);
  const char *f = argv[2];
  [[gnu::cleanup(sqlite_conn_destr)]] sqlite3 *db;
  sqlite3_open(f, &db);
  sqlite_fail_on_err(db);

  [[gnu::cleanup(sqlite_stmt_destr)]] sqlite3_stmt *stmt;
  sqlite3_prepare_v2(db, "select People.name from Scores "
                         "join People on People.id = Scores.player_id "
                         "where round_num = ?1 "
                         "and total = (select max(total) from Scores where round_num = ?1) "
                         "and plus = (select max(plus) from Scores as s2 where round_num = ?1 and s2.total = Scores.total)"
                         "order by player_id asc", -1, &stmt, NULL);
  sqlite_fail_on_err(db);

  sqlite3_bind_int(stmt, 1, roundnum);
  sqlite_fail_on_err(db);  
  unsigned _BitInt(256) hash = sha256hash(argv[1]);

  int ret;
  size_t len = 0;
  size_t cap = 16;
  char **ids = malloc(sizeof(*ids) * cap);
  while ((ret = sqlite3_step(stmt)) == SQLITE_ROW) {
      if (len == cap) {
        cap *= 2;
        ids = realloc(ids, sizeof(*ids) * cap);
      }

      ids[len++] = strdup((char*)sqlite3_column_text(stmt, 0));
  }

  if (ret != SQLITE_DONE)
    sqlite_fail_on_err(db);

  if (len == 0) {
    printf("No candidates found.\n");
    return 1;
  }

  size_t idx = (size_t)(hash % (unsigned _BitInt(256))len);
  printf("Winner : %s!\n", ids[idx]);

  for (size_t i = 0; i < len; ++i) {
    printf("Candidate: %s\n", ids[i]);
    free(ids[i]);
  }

  free(ids);
  
  return 0;
}
