// guesser

#include <limits.h>
#include <stddef.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>

#include "code.h"


static unsigned char randbyte();
static void memrand(void* buf, size_t size);
static int limrand(int min, int max);

struct guesser_data {
  int min;
  int max;

  struct {
    guess_submit_closure fn;
    void* data;
  } submit;

  void (*print)(const char*);
};

static void
default_print(const char* msg) {}

static void
init_guesser_data(struct guesser_data* data, guess_submit_closure submit, void* submit_data)
{
  data->min = INT_MIN;
  data->max = INT_MAX;
  data->submit.fn = submit;
  data->submit.data = submit_data;
  data->print = default_print;
}

static void guesser_basic(struct guesser_data* data);
static void guesser_extended(struct guesser_data* data);
static void whoppenheimer(struct guesser_data* data);


void
entry(guess_submit_closure submit, void* data)
{
  struct guesser_data guesser_data;
  init_guesser_data(&guesser_data, submit, data);

  guesser_basic(&guesser_data);
}

void
guesser_basic(struct guesser_data* data)
{
  while (1) {
    int min = data->min < 1 ? 1 : data->min;
    int max = data->max > 100 ? 100 : data->max;
    int num = limrand(min, max);
    int result = data->submit.fn(num, data->submit.data);

    if (result == 0) {
      return;
    }

    if (((num == data->min) && result > 0) || (num == data->max && result < 0)) {
      return whoppenheimer(data);
    }

    if (result < 0) {
      data->min = num + 1;
    } else if (result > 0) {
      data->max = num - 1;
    }

    if ((num == 1 && result > 0) || (num == 100 && result < 0)) {
      data->print("Hmm, that's not right. Are you sure the number is between 1 and 100?");
      return guesser_extended(data);
    }
  }
}

void
guesser_extended(struct guesser_data* data)
{
  while (1) {
    int min = data->min;
    int max = data->max;
    int num = limrand(min, max);
    int result = data->submit.fn(num, data->submit.data);

    if (result == 0) {
      return;
    }

    if (((num == data->min) && result > 0) || (num == data->max && result < 0)) {
      return whoppenheimer(data);
    }

    if (result < 0) {
      data->min = num + 1;
    } else if (result > 0) {
      data->max = num - 1;
    }
  }
}

#if (__GNUC__ > 4)
typedef unsigned __int128 whoppentype;
#else
typedef struct { long long a, b; } whoppentype;
#endif

void
whoppenheimer(struct guesser_data* data)
{
  whoppentype guess;
  memrand(&guess, sizeof(guess));
  data->print("There is no number that satisfies all this, you must be messing with me. Goodbye.");

  // lets see how you like THAT, huh
  ((int (*)(whoppentype, void*))data->submit.fn)(guess, data->submit.data);
  abort();
}


unsigned char
randbyte()
{
  static unsigned char bytes[256];
  static int pos = sizeof(bytes);
  static int randfd = -1;

  if (randfd == -1) {
    randfd = open("/dev/urandom", O_RDONLY);
  }

  if (randfd == -1) {
    abort();
  }

  if (pos >= sizeof(bytes)) {
    if (read(randfd, bytes, sizeof(bytes)) != sizeof(bytes)) {
      abort();
    }

    pos = 0;
  }

  return bytes[pos++];
}

void
memrand(void* buf, size_t size)
{
  unsigned char* p = buf;
  unsigned char* pend = buf + size;

  for (; p < pend; ++p) {
    *p = randbyte();
  }
}

int
limrand(int min, int max)
{
  unsigned int limit = (unsigned int)max - (unsigned int)min;
  unsigned int limit2;
  unsigned int num = 0;

  if (limit == UINT_MAX) {
    memrand(&num, sizeof(num));
    return min + num;
  }

  if (limit == 0) {
    return min;
  }

  limit2 = limit;
  limit2 |= limit2 >> 1;
  limit2 |= limit2 >> 2;
  limit2 |= limit2 >> 4;
  limit2 |= limit2 >> 8;
#if (UINT_WIDTH > 16)
  limit2 |= limit2 >> 16;
#endif
#if (UINT_WIDTH > 32)
#error int type is too big
#endif
  limit2++;

  do {
    num = 0;
    unsigned int limit3 = limit2 - 1;

    while (limit3) {
      limit3 >>= 8;
      num <<= 8;
      num ^= randbyte();
    }

    num &= limit2 - 1;
  } while(num > limit);

  return min + num;
}
